// Bike Rental Website - Version 3 (HTML/CSS/JS)

// Bike types data
const bikeTypes = [
  { 
    id: 'mountain', 
    name: 'Горный', 
    icon: 'icons/type-mountain.svg', 
    color: '#22C55E',
    description: 'Для горных трасс и бездорожья',
    price: 1500,
    model: 'Trek Mountain Pro'
  },
  { 
    id: 'city', 
    name: 'Городской', 
    icon: 'icons/type-city.svg', 
    color: '#16A34A',
    description: 'Комфортные поездки по городу',
    price: 1200,
    model: 'City Cruiser 2024'
  },
  { 
    id: 'kids', 
    name: 'Детский', 
    icon: 'icons/type-kids.svg', 
    color: '#4ADE80',
    description: 'Безопасные велосипеды для детей',
    price: 800,
    model: 'Kids Bike Junior'
  },
  { 
    id: 'electric', 
    name: 'Электровелосипед', 
    icon: 'icons/type-electric.svg', 
    color: '#22C55E',
    description: 'С электромотором для дальних поездок',
    price: 2000,
    model: 'Electric Speed X1'
  }
]

// Initialize page
function initPage() {
  const currentPage = window.location.pathname.includes('book.html') ? 'book' : 'history'
  
  if (currentPage === 'book') {
    initBookPage()
  } else {
    initHistoryPage()
  }
  
  initSlider()
}

// Initialize history page
function initHistoryPage() {
  loadRentals()
  updateStats()
  
  // Tab switching
  const tabs = document.querySelectorAll('.tab-button')
  tabs.forEach(tab => {
    tab.addEventListener('click', function() {
      tabs.forEach(t => t.classList.remove('active'))
      this.classList.add('active')
      const tabType = this.dataset.tab
      renderRentals(tabType)
    })
  })
  
  // Initial render
  renderRentals('active')
  
  // Poll for storage changes
  setInterval(() => {
    loadRentals()
    updateStats()
  }, 500)
}

// Load rentals from localStorage
function loadRentals() {
  const saved = localStorage.getItem('bikeRentals')
  let rentals = { active: [], past: [] }
  
  if (saved) {
    try {
      rentals = JSON.parse(saved)
    } catch (e) {
      rentals = { active: [], past: [] }
    }
  } else {
    // Default rentals
    rentals = {
      active: [
        {
          id: 1,
          model: 'Trek Mountain Pro',
          type: 'Горный',
          date: '2025-03-28',
          returnDate: '2025-04-05',
          price: 1500,
          status: 'active'
        },
        {
          id: 2,
          model: 'City Cruiser 2024',
          type: 'Городской',
          date: '2025-03-25',
          returnDate: '2025-04-02',
          price: 1200,
          status: 'active'
        }
      ],
      past: [
        {
          id: 3,
          model: 'Electric Speed X1',
          type: 'Электровелосипед',
          date: '2025-03-15',
          returnDate: '2025-03-20',
          price: 2000,
          review: 'Отличный велосипед, очень удобный!'
        },
        {
          id: 4,
          model: 'Kids Bike Junior',
          type: 'Детский',
          date: '2025-03-10',
          returnDate: '2025-03-12',
          price: 800,
          review: 'Ребенок в восторге!'
        }
      ]
    }
    localStorage.setItem('bikeRentals', JSON.stringify(rentals))
  }
  
  return rentals
}

// Update stats
function updateStats() {
  const rentals = loadRentals()
  const stats = [
    { 
      value: rentals.past.length, 
      label: 'Завершено аренд',
      icon: 'icons/stat-bike.svg',
      color: '#22C55E'
    },
    { 
      value: rentals.past.reduce((sum, r) => sum + (r.price || 0), 0), 
      label: 'Потрачено',
      icon: 'icons/stat-dollar.svg',
      color: '#16A34A'
    },
    { 
      value: rentals.past.filter(r => r.review).length, 
      label: 'Оставлено отзывов',
      icon: 'icons/stat-message.svg',
      color: '#4ADE80'
    }
  ]
  
  const statsGrid = document.getElementById('statsGrid')
  if (statsGrid) {
    statsGrid.innerHTML = stats.map(stat => `
      <div class="sidebar-stat-card">
        <div class="sidebar-stat-icon" style="background: ${stat.color}20">
          <img src="${stat.icon}" alt="" class="stat-icon" width="24" height="24" />
        </div>
        <div class="sidebar-stat-content">
          <div class="sidebar-stat-value">
            ${stat.label === 'Потрачено' ? `${stat.value} ₽` : stat.value}
          </div>
          <div class="sidebar-stat-label">${stat.label}</div>
        </div>
      </div>
    `).join('')
  }

  // Update counts
  const activeCount = document.getElementById('activeCount')
  const pastCount = document.getElementById('pastCount')
  if (activeCount) activeCount.textContent = rentals.active.length
  if (pastCount) pastCount.textContent = rentals.past.length
}

// Render rentals
function renderRentals(type) {
  const rentals = loadRentals()
  const currentRentals = rentals[type]
  const rentalsList = document.getElementById('rentalsList')
  
  if (!rentalsList) return
  
  if (currentRentals.length === 0) {
    rentalsList.innerHTML = `
      <div class="empty-state">
        <img src="icons/rental-bike.svg" alt="" class="empty-icon" width="64" height="64" />
        <p>${type === 'active' ? 'У вас нет активных аренд' : 'История аренд пуста'}</p>
      </div>
    `
    return
  }
  
  rentalsList.innerHTML = currentRentals.map(rental => {
    const startDate = new Date(rental.date).toLocaleDateString('ru-RU', { day: 'numeric', month: 'short' })
    const endDate = new Date(rental.returnDate).toLocaleDateString('ru-RU', { day: 'numeric', month: 'short' })
    
    return `
      <div class="rental-item">
        <div class="rental-item-left">
          <div class="rental-item-icon">
            <img src="icons/rental-bike.svg" alt="" width="32" height="32" />
          </div>
          <div class="rental-item-info">
            <h3 class="rental-item-model">${rental.model}</h3>
            <p class="rental-item-type">${rental.type}</p>
            ${rental.review ? `
              <div class="rental-item-review">
                <img src="icons/stat-message.svg" alt="" width="14" height="14" />
                <span>${rental.review}</span>
              </div>
            ` : ''}
          </div>
        </div>
        <div class="rental-item-center">
          <div class="rental-item-date">
            <img src="icons/form-calendar.svg" alt="" width="16" height="16" />
            <span>${startDate} - ${endDate}</span>
          </div>
        </div>
        <div class="rental-item-right">
          <div class="rental-item-price">${rental.price} ₽</div>
          ${rental.status === 'active' ? '<span class="rental-item-status">Активна</span>' : ''}
        </div>
      </div>
    `
  }).join('')
}

// Initialize book page
function initBookPage() {
  renderBikeTypes()
  
  const form = document.getElementById('bookingForm')
  const returnDateInput = document.getElementById('returnDate')
  const reviewInput = document.getElementById('review')
  let selectedType = ''
  
  // Set date limits
  const today = new Date()
  const maxDate = new Date(today.getTime() + 30 * 24 * 60 * 60 * 1000)
  returnDateInput.min = today.toISOString().split('T')[0]
  returnDateInput.max = maxDate.toISOString().split('T')[0]
  
  // Type selection
  document.querySelectorAll('.type-button').forEach(button => {
    button.addEventListener('click', function() {
      document.querySelectorAll('.type-button').forEach(b => b.classList.remove('selected'))
      this.classList.add('selected')
      selectedType = this.dataset.type
      updatePriceSummary()
    })
  })
  
  // Date change
  returnDateInput.addEventListener('change', updatePriceSummary)
  
  // Form submit
  form.addEventListener('submit', function(e) {
    e.preventDefault()
    
    if (!selectedType || !returnDateInput.value) {
      alert('Пожалуйста, выберите тип велосипеда и дату возврата')
      return
    }
    
    const savedRentals = localStorage.getItem('bikeRentals')
    const rentals = savedRentals ? JSON.parse(savedRentals) : { active: [], past: [] }
    
    const bikeType = bikeTypes.find(t => t.id === selectedType)
    const newRental = {
      id: Date.now(),
      model: bikeType.model,
      type: bikeType.name,
      date: today.toISOString().split('T')[0],
      returnDate: returnDateInput.value,
      price: bikeType.price,
      status: 'active',
      review: reviewInput.value || null
    }
    
    rentals.active.push(newRental)
    localStorage.setItem('bikeRentals', JSON.stringify(rentals))
    
    // Show success message
    const successMsg = document.getElementById('successMessage')
    successMsg.style.display = 'flex'
    
    setTimeout(() => {
      window.location.href = 'index.html'
    }, 2000)
  })
}

// Render bike types
function renderBikeTypes() {
  const typesRow = document.getElementById('typesRow')
  if (!typesRow) return
  
  typesRow.innerHTML = bikeTypes.map(type => `
    <button type="button" class="type-button" data-type="${type.id}">
      <div class="type-button-icon">
        <img src="${type.icon}" alt="${type.name}" width="32" height="32" />
      </div>
      <div class="type-button-content">
        <h3 class="type-button-name">${type.name}</h3>
        <p class="type-button-price">${type.price} ₽/день</p>
      </div>
    </button>
  `).join('')
}

// Update price summary
function updatePriceSummary() {
  const selectedButton = document.querySelector('.type-button.selected')
  const returnDateInput = document.getElementById('returnDate')
  const priceSummary = document.getElementById('priceSummary')
  
  if (!selectedButton || !returnDateInput.value || !priceSummary) return
  
  const typeId = selectedButton.dataset.type
  const bikeType = bikeTypes.find(t => t.id === typeId)
  if (!bikeType) return
  
  const today = new Date()
  const returnDate = new Date(returnDateInput.value)
  const days = Math.ceil((returnDate - today) / (1000 * 60 * 60 * 24))
  const total = bikeType.price * days
  
  document.getElementById('summaryPrice').textContent = `${bikeType.price} ₽`
  document.getElementById('summaryDays').textContent = `${days} дней`
  document.getElementById('totalPrice').textContent = `${total} ₽`
  
  priceSummary.style.display = 'block'
}

// Initialize slider
function initSlider() {
  const slider = document.getElementById('slider')
  if (!slider) return
  
  const slides = slider.querySelectorAll('.slide')
  const dots = slider.querySelectorAll('.dot')
  const prevBtn = document.getElementById('sliderPrev')
  const nextBtn = document.getElementById('sliderNext')
  
  let currentIndex = 0
  
  function showSlide(index) {
    slides.forEach((slide, i) => {
      slide.classList.toggle('active', i === index)
    })
    dots.forEach((dot, i) => {
      dot.classList.toggle('active', i === index)
    })
  }
  
  function nextSlide() {
    currentIndex = (currentIndex + 1) % slides.length
    showSlide(currentIndex)
  }
  
  function prevSlide() {
    currentIndex = (currentIndex - 1 + slides.length) % slides.length
    showSlide(currentIndex)
  }
  
  if (prevBtn) prevBtn.addEventListener('click', prevSlide)
  if (nextBtn) nextBtn.addEventListener('click', nextSlide)
  
  dots.forEach((dot, index) => {
    dot.addEventListener('click', () => {
      currentIndex = index
      showSlide(currentIndex)
    })
  })
  
  // Auto-advance every 3 seconds
  setInterval(nextSlide, 3000)
}

// Initialize on page load
document.addEventListener('DOMContentLoaded', initPage)

